### Variance estimation for low income proportion: Example ###

### Load packages ###############################################################################################################

  library(foreign)
  library(Hmisc)
  library(samplingVarEst)
  
### General setup ###############################################################################################################  

  scale <- 0.54 # Scale value
  scales_a <- seq(0.32,0.72,by=0.005) # Distribution of scales A (Unif~[0.32,0.72])
  scales_b <- seq(0.34,0.51,by=0.002) # Distribution of scales B (Unif~[0.34,0.51])
  beta <- 0.5 # Which quintile for poverty line
  alpha <- 0.6 # Which fraction of quintile for poverty line
  qx <- 0.5 # Quantile of interest (in addition to min and max)
  years <- 2000:2012 # Years
  n_years <- length(years)
  soepcodes <- c("a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z",
                 "ba","bb","bc")
  soepyears <- 1984:2012

### Objects for results #########################################################################################################

  ### Number of obs

  sample_sizes <- numeric(n_years)
  population_sizes <- numeric(n_years)

  ### Standard estimates + inference ##########################################

  # Estimates
  lip <- numeric(n_years) # Low income proportion
  lip.w <- numeric(n_years) # Low income proportion using raking weights

  # Variance
  s_var <- numeric(n_years) # Variance estimate of LIP (object lip)
  s_var.raking <- numeric(n_years) # Variance of weighted LIP, only accounting for raking (object lip.w)
  s_var.raking.design <- numeric(n_years) # Variance of weighted LIP, accounting for raking and design weights (object lip.w)
  s_var.raking.designyear <- numeric(n_years) # Variance of weighted LIP, accounting for raking and synthetic design weights (object lip.w)

  # Confidence intervals
  CI <- matrix(data=NA,ncol=2,nrow=n_years) # Confidence interval of low income proportion (object lip)
  CI.raking <- matrix(data=NA,ncol=2,nrow=n_years) # Confidence interval of weighted low income proportion, only raking (object lip.w)
  CI.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # Confidence interval of weighted low income proportion, raking+design (object lip.w)
  CI.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # Confidence interval of weighted low income proportion, raking+synthetic design (object lip.w)

  ### Estimates and inference on MEAN of induced distribution ###################

  # Estimates on distribution A
  lip_a.mean <- numeric(n_years) # Mean of low income proportion
  lip_a.weighted.mean <- numeric(n_years) # Mean of weighted low income proportion

  # Variance of estimates on distribution A
  s_var_a <- numeric(n_years) # Variance estimate of mean of induced distribution (object lip_a.mean)
  s_var_a.raking <- numeric(n_years) # Variance estimate of mean of induced distribution, only accounting for raking (object lip_a.weighted.mean)
  s_var_a.raking.design <- numeric(n_years) # Variance estimate of mean of induced distribution, raking+design (object lip_a.weighted.mean)
  s_var_a.raking.designyear <- numeric(n_years) # Variance estimate of mean of induced distribution, raking+synthetic design (object lip_a.weighted.mean)

  # CIs of estimates on distribution A
  CI_a <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution (object lip_a.mean)
  CI_a.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution, only accounting for raking (object lip_a.weighted.mean)
  CI_a.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution,accounting for raking and design (object lip_a.weighted.mean)
  CI_a.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution, accounting for rakind and synthetic design (object lip_a.weighted.mean)

  # Estimates on distribution B 
  lip_b.mean <- numeric(n_years) # Mean of low income proportion
  lip_b.weighted.mean <- numeric(n_years) # Mean of weighted low income proportion

  # Variance of estimates on distribution B
  s_var_b <- numeric(n_years) # Variance estimate of mean of induced distribution (object lip_b.mean)
  s_var_b.raking <- numeric(n_years) # Variance estimate of mean of induced distribution, only accounting for raking (object lip_b.weighted.mean)
  s_var_b.raking.design <- numeric(n_years) # Variance estimate of mean of induced distribution, raking+design (object lip_b.weighted.mean)
  s_var_b.raking.designyear <- numeric(n_years) # Variance estimate of mean of induced distribution, raking+synthetic design (object lip_b.weighted.mean)

  # # CIs of estimates on distribution B
  CI_b <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution (object lip_b.mean)
  CI_b.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution, only accounting for raking (object lip_b.weighted.mean)
  CI_b.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution,accounting for raking and design (object lip_b.weighted.mean)
  CI_b.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of mean of induced distribution, accounting for rakind and synthetic design (object lip_b.weighted.mean)

  ### Estimates and inference on MEDIAN of induced distribution ###################

  # Estimates on distribution A
  lip_a.q <- numeric(n_years) # Median of induced distribution
  lip_a.weighted.q <- numeric(n_years) # Weighted median of induced distribution

  # Variance of estimates on distribution A
  s_var_a.q <- numeric(n_years) # Variance of median of induced distribution (object lip_a.q)
  s_var_a.q.raking <- numeric(n_years) # Variance of weighted median of induced distribution only accounting for raking (object lip_a.weighted.q)
  s_var_a.q.raking.design <- numeric(n_years) # Variance of weighted median of induced distribution accounting for raking+design (object lip_a.weighted.q)
  s_var_a.q.raking.designyear <- numeric(n_years) # Variance of weighted median of induced distribution accounting for raking+synthetic design (object lip_a.weighted.q)

  # CIs of estimates on distribution A
  CI_a.q <- matrix(data=NA,ncol=2,nrow=n_years) # CI of median of induced distribution (object lip_a.q)
  CI_a.q.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted median of induced distribution only accounting for raking (object lip_a.weighted.q)
  CI_a.q.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted median of induced distribution accounting for raking+design (object lip_a.weighted.q)
  CI_a.q.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted median of induced distribution accounting for raking+synthetic design (object lip_a.weighted.q)

  # Estimates on distribution B
  lip_b.q <- numeric(n_years) # Median of induced distribution
  lip_b.weighted.q <- numeric(n_years) # Weighted median of induced distribution
  
  # Variance of estimates on distribution B
  s_var_b.q <- numeric(n_years) # Variance of median of induced distribution (object lip_b.q)
  s_var_b.q.raking <- numeric(n_years) # Variance of weighted median of induced distribution only accounting for raking (object lip_b.weighted.q)
  s_var_b.q.raking.design <- numeric(n_years) # Variance of weighted median of induced distribution accounting for raking+design (object lip_b.weighted.q)
  s_var_b.q.raking.designyear <- numeric(n_years) # Variance of weighted median of induced distribution accounting for raking+synthetic design (object lip_b.weighted.q)
  
  # CIs of estimates on distribution B
  CI_b.q <- matrix(data=NA,ncol=2,nrow=n_years) # CI of median of induced distribution (object lip_b.q)
  CI_b.q.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted median of induced distribution only accounting for raking (object lip_b.weighted.q)
  CI_b.q.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted median of induced distribution accounting for raking+design (object lip_b.weighted.q)
  CI_b.q.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted median of induced distribution accounting for raking+synthetic design (object lip_b.weighted.q)

  ### Estimates and inference on MINIMUM of induced distribution ###################
  
  # Estimates on distribution A
  lip_a.min <- numeric(n_years) # Minimum of induced distribution
  lip_a.weighted.min <- numeric(n_years) # Weighted minimum of induced distribution
  
  # Variance of estimates on distribution A
  s_var_a.min <- numeric(n_years) # Variance of minimum of induced distribution (object lip_a.min)
  s_var_a.min.raking <- numeric(n_years) # Variance of weighted minimum of induced distribution only accounting for raking (object lip_a.weighted.min)
  s_var_a.min.raking.design <- numeric(n_years) # Variance of weighted minimum of induced distribution accounting for raking+design (object lip_a.weighted.min)
  s_var_a.min.raking.designyear <- numeric(n_years) # Variance of weighted minimum of induced distribution accounting for raking+synthetic design (object lip_a.weighted.min)
  
  # CIs of estimates on distribution A
  CI_a.min <- matrix(data=NA,ncol=2,nrow=n_years) # CI of minimum of induced distribution (object lip_a.min)
  CI_a.min.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted minimum of induced distribution only accounting for raking (object lip_a.weighted.min)
  CI_a.min.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted minimum of induced distribution accounting for raking+design (object lip_a.weighted.min)
  CI_a.min.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted minimum of induced distribution accounting for raking+synthetic design (object lip_a.weighted.min)
  
  # Estimates on distribution B
  lip_b.min <- numeric(n_years) # minimum of induced distribution
  lip_b.weighted.min <- numeric(n_years) # Weighted minimum of induced distribution
  
  # Variance of estimates on distribution B
  s_var_b.min <- numeric(n_years) # Variance of minimum of induced distribution (object lip_b.min)
  s_var_b.min.raking <- numeric(n_years) # Variance of weighted minimum of induced distribution only accounting for raking (object lip_b.weighted.min)
  s_var_b.min.raking.design <- numeric(n_years) # Variance of weighted minimum of induced distribution accounting for raking+design (object lip_b.weighted.min)
  s_var_b.min.raking.designyear <- numeric(n_years) # Variance of weighted minimum of induced distribution accounting for raking+synthetic design (object lip_b.weighted.min)
  
  # CIs of estimates on distribution B
  CI_b.min <- matrix(data=NA,ncol=2,nrow=n_years) # CI of minimum of induced distribution (object lip_b.min)
  CI_b.min.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted minimum of induced distribution only accounting for raking (object lip_b.weighted.min)
  CI_b.min.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted minimum of induced distribution accounting for raking+design (object lip_b.weighted.min)
  CI_b.min.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted minimum of induced distribution accounting for raking+synthetic design (object lip_b.weighted.min)

  ### Estimates and inference on MAXIMUM of induced distribution ###################
  
  # Estimates on distribution A
  lip_a.max <- numeric(n_years) # Maximum of induced distribution
  lip_a.weighted.max <- numeric(n_years) # Weighted maximum of induced distribution
  
  # Variance of estimates on distribution A
  s_var_a.max <- numeric(n_years) # Variance of maximum of induced distribution (object lip_a.max)
  s_var_a.max.raking <- numeric(n_years) # Variance of weighted maximum of induced distribution only accounting for raking (object lip_a.weighted.max)
  s_var_a.max.raking.design <- numeric(n_years) # Variance of weighted maximum of induced distribution accounting for raking+design (object lip_a.weighted.max)
  s_var_a.max.raking.designyear <- numeric(n_years) # Variance of weighted maximum of induced distribution accounting for raking+synthetic design (object lip_a.weighted.max)
  
  # CIs of estimates on distribution A
  CI_a.max <- matrix(data=NA,ncol=2,nrow=n_years) # CI of maximum of induced distribution (object lip_a.max)
  CI_a.max.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted maximum of induced distribution only accounting for raking (object lip_a.weighted.max)
  CI_a.max.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted maximum of induced distribution accounting for raking+design (object lip_a.weighted.max)
  CI_a.max.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted maximum of induced distribution accounting for raking+synthetic design (object lip_a.weighted.max)
  
  # Estimates on distribution B
  lip_b.max <- numeric(n_years) # maximum of induced distribution
  lip_b.weighted.max <- numeric(n_years) # Weighted maximum of induced distribution
  
  # Variance of estimates on distribution B
  s_var_b.max <- numeric(n_years) # Variance of maximum of induced distribution (object lip_b.max)
  s_var_b.max.raking <- numeric(n_years) # Variance of weighted maximum of induced distribution only accounting for raking (object lip_b.weighted.max)
  s_var_b.max.raking.design <- numeric(n_years) # Variance of weighted maximum of induced distribution accounting for raking+design (object lip_b.weighted.max)
  s_var_b.max.raking.designyear <- numeric(n_years) # Variance of weighted maximum of induced distribution accounting for raking+synthetic design (object lip_b.weighted.max)
  
  # CIs of estimates on distribution B
  CI_b.max <- matrix(data=NA,ncol=2,nrow=n_years) # CI of maximum of induced distribution (object lip_b.max)
  CI_b.max.raking <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted maximum of induced distribution only accounting for raking (object lip_b.weighted.max)
  CI_b.max.raking.design <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted maximum of induced distribution accounting for raking+design (object lip_b.weighted.max)
  CI_b.max.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years) # CI of weighted maximum of induced distribution accounting for raking+synthetic design (object lip_b.weighted.max)

  ### Joint CI Min/Max ########################################################

  # Distribution A
  CI_a.joint <- matrix(data=NA,ncol=2,nrow=n_years)
  CI_a.joint.raking <- matrix(data=NA,ncol=2,nrow=n_years)
  CI_a.joint.raking.design <- matrix(data=NA,ncol=2,nrow=n_years)
  CI_a.joint.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years)

  # Distribution B
  CI_b.joint <- matrix(data=NA,ncol=2,nrow=n_years)
  CI_b.joint.raking <- matrix(data=NA,ncol=2,nrow=n_years)
  CI_b.joint.raking.design <- matrix(data=NA,ncol=2,nrow=n_years)
  CI_b.joint.raking.designyear <- matrix(data=NA,ncol=2,nrow=n_years)


### Begin analysis loop #########################################################################################################

  for(j in years) {
    
    nummer <- which(years==j)
    cat(j,"   ",date(), "\n")

### Read data ###################################################################################################################

  # Load data
  datei <- paste("/home/christian/Dokumente/Daten/equivar/example",j,".dta",sep="")
  dat <- read.dta(datei,convert.factors=FALSE)
  # Only keep variables of interest
  jahrkurz <- substr(as.character(j),3,4)
  code <- soepcodes[which(soepyears==j)]
  dat <- dat[,c(paste("d11106",jahrkurz,sep=""),
                paste("d11107",jahrkurz,sep=""),
                paste("i11102",jahrkurz,sep=""),
                paste("l11102",jahrkurz,sep=""),
                paste("h11101",jahrkurz,sep=""),
                "design",
                paste("design",j,sep=""),
                paste(code,"hhrf",sep=""),
                paste("owner",jahrkurz,sep=""),
                paste(code,"bula",sep=""),
                paste(code,"ggk",sep=""))] 
  # Rename variables
  names(dat) <- c("family.size","no.children","income","ow","no.child.14","design","designyear","raking","owner","bula","comsize") 

### Pop./sample size + rescaling ################################################################################################
  
  # Sizes
  N <- sum(dat$raking)
  n <- dim(dat)[1]

  population_sizes[nummer] <- N
  sample_sizes[nummer] <- n

  # Rescale design weights
  dat$design <- dat$design * sum(dat$raking)/sum(dat$design)
  dat$designyear <- dat$designyear * sum(dat$raking)/sum(dat$designyear,na.rm=T)
  
### Preliminaries ###############################################################################################################  
  
  # Calculate Low Income proportion (Random sample)

    # Equivalized income
    dat$equiv.weight <- dat$family.size^scale
    dat$equiv.inc <- dat$income/dat$equiv.weight
    # Poverty line 
    poverty.line <- alpha*quantile(dat$equiv.inc,probs=beta) 
    # Poverty indicator 
    dat$poor <- 0
    dat$poor[dat$equiv.inc<=poverty.line] <- 1
    # Low income proportion 
    lip[nummer] <- mean(dat$poor)
  
  # Calculate Low Income Proportion (Weighted/Raking)
  
    # Poverty line
    poverty.line.w <- alpha*wtd.quantile(dat$equiv.inc,weights=dat$raking,probs=beta) 
    # Poverty indicator 
    dat$poor <- 0
    dat$poor[dat$equiv.inc<=poverty.line.w] <- 1
    # Low income proportion 
    lip.w[nummer] <- sum((dat$poor*dat$raking)/sum(dat$raking))
  
### Standard Inference ##########################################################################################################  
  
  # Calculate IF random sample
  
    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (quantile(dat$equiv.inc,probs=0.75)-quantile(dat$equiv.inc,probs=0.25))* n^(-0.2)
    tmp <- density(dat$equiv.inc,bw=bandwidth)
    f1 <- tmp$y[which.min(abs(tmp$x-poverty.line))]
    f2 <- tmp$y[which.min(abs(tmp$x-poverty.line/alpha))]
  
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat$equiv.inc<=median(dat$equiv.inc)] <- 1
    dat$z1 <- 1/N * (dat$poor - lip[nummer])
    dat$z2 <- -alpha * 1/N * (f1/f2) * (dat$below.median-beta)
    dat$z <- dat$z1+dat$z2
    # Variance estimate
    s_var[nummer] <- var(dat$z)*(N*(N-n))/n 
  
    # CI
    CI[nummer,] <- c ( lip[nummer]+1.96*sqrt(s_var[nummer]) ,  lip[nummer]-1.96*sqrt(s_var[nummer]) )
  
  # Weighted, only accounting for raking

    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (wtd.quantile(dat$equiv.inc,weights=dat$raking,probs=0.75)-wtd.quantile(dat$equiv.inc,weights=dat$raking,probs=0.25))* n^(-0.2)
    tmp <- density(dat$equiv.inc,bw=bandwidth,weights=dat$raking/sum(dat$raking))
    f1 <- tmp$y[which.min(abs(tmp$x-poverty.line.w))]
    f2 <- tmp$y[which.min(abs(tmp$x-poverty.line.w/alpha))]
  
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat$equiv.inc<=wtd.quantile(dat$equiv.inc,weights=dat$raking,probs=0.5)] <- 1
    dat$poor <- 0
    dat$poor[dat$equiv.inc<=poverty.line.w] <- 1
    dat$z1 <- 1/N * (dat$poor - lip.w[nummer])
    dat$z2 <- -alpha * 1/N * (f1/f2) * (dat$below.median-beta)
    dat$z <- dat$z1+dat$z2
  
    # Correction of z
    fit.raking <- lm(z~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
    dat$z <- fit.raking$residuals
  
    # Variance estimate
    s_var.raking[nummer] <- var(dat$z)*(N*(N-n))/n 
  
    # CI
    CI.raking[nummer,] <- c ( lip.w[nummer]+1.96*sqrt(s_var.raking[nummer]) ,  lip.w[nummer]-1.96*sqrt(s_var.raking[nummer]) )
  
  # Variance estimate using design weights
  
    # Tmp Dummy (because some design weights=0)
    dat$use <- 0
    dat$use[dat$design>0] <- 1
  
    # Variance estimate  
    s_var.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  
    # CI
    CI.raking.design[nummer,] <- c ( lip.w[nummer]+1.96*sqrt(s_var.raking.design[nummer]) ,  lip.w[nummer]-1.96*sqrt(s_var.raking.design[nummer]) )
  
  # Variance estimate using synthetic design weights
  
    # Tmp dummy (because some synthetic design weights=0)
    dat$use <- 0
    dat$use[dat$designyear>0] <- 1
  
    # Variance estimate
    s_var.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])

    # CI
    CI.raking.designyear[nummer,] <- c ( lip.w[nummer]+1.96*sqrt(s_var.raking.designyear[nummer]) ,  lip.w[nummer]-1.96*sqrt(s_var.raking.designyear[nummer]) )
  
### Descriptives for induced distributions ######################################################################################
### (And Influence functions for inference on mean ...)
  
  # Induced distribution variant A (broad range), not weighted
  lips_a <- numeric(length(scales_a))
  for(i in 1:length(scales_a)) {
    dat$equiv.weight <- dat$family.size^scales_a[i]
    dat[,paste("equiv.inc",i,sep=".")] <- dat$income/dat$equiv.weight
    # Poverty line 
    poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",i,sep=".")],probs=beta) 
    # Poverty indicator 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",i,sep=".")]<=poverty.line.tmp] <- 1
    # Low income proportion 
    lips_a[i] <- mean(dat$poor)
    
    bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",i,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",i,sep=".")],probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",i,sep=".")],bw=bandwidth)
    f1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    f2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",i,sep=".")]<=median(dat[,paste("equiv.inc",i,sep=".")])] <- 1
    dat$z1 <- 1/N * (dat$poor - lips_a[i])
    dat$z2 <- -alpha * 1/N * (f1/f2) * (dat$below.median-beta)
    dat[,paste("zz",i,sep="")]  <- dat$z1+dat$z2
  }
  
  # IF mean
  dat$z_a <- rowMeans(dat[,paste("zz",1:length(scales_a),sep="")])
  
  # Mean and quantiles
  lip_a.mean[nummer] <- mean(lips_a)
  lip_a.q[nummer] <- quantile(lips_a,probs=qx)
  lip_a.min[nummer] <- min(lips_a)
  lip_a.max[nummer] <- max(lips_a)
  
  # Induced distribution variant A (broad range), WEIGHTED
  lips_a.weighted <- numeric(length(scales_a))
  for(i in 1:length(scales_a)) {
    dat$equiv.weight <- dat$family.size^scales_a[i]
    dat[,paste("equiv.inc",i,sep=".")] <- dat$income/dat$equiv.weight
    # Poverty line 
    poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=beta) 
    # Poverty indicator 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",i,sep=".")]<=poverty.line.tmp] <- 1
    # Low income proportion 
    lips_a.weighted[i] <- sum((dat$poor*dat$raking)/sum(dat$raking))
    
    bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",i,sep=".")],bw=bandwidth,weights=dat$raking/sum(dat$raking))
    f1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    f2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",i,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=0.5)] <- 1
    dat$z1 <- 1/N * (dat$poor - lips_a.weighted[i])
    dat$z2 <- -alpha * 1/N * (f1/f2) * (dat$below.median-beta)
    dat[,paste("zz",i,sep="")]  <- dat$z1+dat$z2
    
  }
  
  # IF mean weighted
  dat$z_a.weighted <- rowMeans(dat[,paste("zz",1:length(scales_a),sep="")])
  
  # Mean and quantiles
  lip_a.weighted.mean[nummer] <- mean(lips_a.weighted)
  lip_a.weighted.q[nummer] <- quantile(lips_a.weighted,probs=qx)
  lip_a.weighted.min[nummer] <- min(lips_a.weighted)
  lip_a.weighted.max[nummer] <- max(lips_a.weighted)
  
  # Induced distribution variant B (expenditure scales), not weighted
  lips_b <- numeric(length(scales_b))
  for(i in 1:length(scales_b)) {
    dat$equiv.weight <- dat$family.size^scales_b[i]
    dat[,paste("equiv.inc",i,sep=".")] <- dat$income/dat$equiv.weight
    # Poverty line 
    poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",i,sep=".")],probs=beta) 
    # Poverty indicator 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",i,sep=".")]<=poverty.line.tmp] <- 1
    # Low income proportion 
    lips_b[i] <- mean(dat$poor)
    
    bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",i,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",i,sep=".")],probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",i,sep=".")],bw=bandwidth)
    f1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    f2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",i,sep=".")]<=median(dat[,paste("equiv.inc",i,sep=".")])] <- 1
    dat$z1 <- 1/N * (dat$poor - lips_b[i])
    dat$z2 <- -alpha * 1/N * (f1/f2) * (dat$below.median-beta)
    dat[,paste("zz",i,sep="")]  <- dat$z1+dat$z2
    
  }
  
  # IF mean
  dat$z_b <- rowMeans(dat[,paste("zz",1:length(scales_b),sep="")])
    
  # Mean and quantiles
  lip_b.mean[nummer] <- mean(lips_b)
  lip_b.q[nummer] <- quantile(lips_b,probs=qx)
  lip_b.min[nummer] <- min(lips_b)
  lip_b.max[nummer] <- max(lips_b)
  
  # Induced distribution variant B (expenditure scales), WEIGHTED
  lips_b.weighted <- numeric(length(scales_b))
  for(i in 1:length(scales_b)) {
    dat$equiv.weight <- dat$family.size^scales_b[i]
    dat[,paste("equiv.inc",i,sep=".")] <- dat$income/dat$equiv.weight
    # Poverty line 
    poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=beta) 
    # Poverty indicator 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",i,sep=".")]<=poverty.line.tmp] <- 1
    # Low income proportion 
    lips_b.weighted[i] <- sum((dat$poor*dat$raking)/sum(dat$raking))
    
    bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",i,sep=".")],bw=bandwidth,weights=dat$raking/sum(dat$raking))
    f1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    f2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",i,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",i,sep=".")],weights=dat$raking,probs=0.5)] <- 1
    dat$z1 <- 1/N * (dat$poor - lips_b.weighted[i])
    dat$z2 <- -alpha * 1/N * (f1/f2) * (dat$below.median-beta)
    dat[,paste("zz",i,sep="")]  <- dat$z1+dat$z2
    
  }
  
  # IF mean weighted
  dat$z_b.weighted <- rowMeans(dat[,paste("zz",1:length(scales_b),sep="")])
  
  # Mean and quantiles
  lip_b.weighted.mean[nummer] <- mean(lips_b.weighted)
  lip_b.weighted.q[nummer] <- quantile(lips_b.weighted,probs=qx)
  lip_b.weighted.min[nummer] <- min(lips_b.weighted)
  lip_b.weighted.max[nummer] <- max(lips_b.weighted)
  
### Variance estimates and CIs for mean of induced distribution #################################################################
  
  # Variant A (Random Sample)
  s_var_a[nummer] <- var(dat$z_a)*(N*(N-n))/n 
  # CI
  CI_a[nummer,] <- c ( lip_a.mean[nummer]+1.96*sqrt(s_var_a[nummer]) ,  lip_a.mean[nummer]-1.96*sqrt(s_var_a[nummer]) )
  
  # Variant A (Weighted; accounting only for raking)
  # Correction of z
  fit.raking <- lm(z_a.weighted~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$z_a.weighted <- fit.raking$residuals
  # Variance estimate
  s_var_a.raking[nummer] <- var(dat$z_a.weighted)*(N*(N-n))/n 
  # CI
  CI_a.raking[nummer,] <- c ( lip_a.weighted.mean[nummer]+1.96*sqrt(s_var_a.raking[nummer]) ,  lip.w[nummer]-1.96*sqrt(s_var_a.raking[nummer]) )
  
  # Variant A design weights
  # Tmp Dummy (because some design weights=0)
  dat$use <- 0
  dat$use[dat$design>0] <- 1
  # Variance estimate  
  s_var_a.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z_a.weighted[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  # CI
  CI_a.raking.design[nummer,] <- c ( lip_a.weighted.mean[nummer]+1.96*sqrt(s_var_a.raking.design[nummer]) ,  lip_a.weighted.mean[nummer]-1.96*sqrt(s_var_a.raking.design[nummer]) )
  
  # Variant A synthetic design weights
  # Tmp dummy (because some synthetic design weights=0)
  dat$use <- 0
  dat$use[dat$designyear>0] <- 1
  # Variance estimate
  s_var_a.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z_a.weighted[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  # CI
  CI_a.raking.designyear[nummer,] <- c ( lip_a.weighted.mean[nummer]+1.96*sqrt(s_var_a.raking.designyear[nummer]) ,  lip_a.weighted.mean[nummer]-1.96*sqrt(s_var_a.raking.designyear[nummer]) )
  
  # Variant B (Random Sample)
  s_var_b[nummer] <- var(dat$z_b)*(N*(N-n))/n 
  # CI
  CI_b[nummer,] <- c ( lip_b.mean[nummer]+1.96*sqrt(s_var_b[nummer]) ,  lip_b.mean[nummer]-1.96*sqrt(s_var_b[nummer]) )
  
  # Variant B (Weighted; accounting only for raking)
  # Correction of z
  fit.raking <- lm(z_b.weighted~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$z_b.weighted <- fit.raking$residuals
  # Variance estimate
  s_var_b.raking[nummer] <- var(dat$z_b.weighted)*(N*(N-n))/n 
  # CI
  CI_b.raking[nummer,] <- c ( lip_b.weighted.mean[nummer]+1.96*sqrt(s_var_b.raking[nummer]) ,  lip.w[nummer]-1.96*sqrt(s_var_b.raking[nummer]) )
  
  # Variant B design weights
  # Tmp Dummy (because some design weights=0)
  dat$use <- 0
  dat$use[dat$design>0] <- 1
  # Variance estimate  
  s_var_b.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z_b.weighted[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  # CI
  CI_b.raking.design[nummer,] <- c ( lip_b.weighted.mean[nummer]+1.96*sqrt(s_var_b.raking.design[nummer]) ,  lip_b.weighted.mean[nummer]-1.96*sqrt(s_var_b.raking.design[nummer]) )
  
  # Variant B synthetic design weights
  # Tmp dummy (because some synthetic design weights=0)
  dat$use <- 0
  dat$use[dat$designyear>0] <- 1
  # Variance estimate
  s_var_b.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z_b.weighted[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  # CI
  CI_b.raking.designyear[nummer,] <- c ( lip_b.weighted.mean[nummer]+1.96*sqrt(s_var_b.raking.designyear[nummer]) ,  lip_b.weighted.mean[nummer]-1.96*sqrt(s_var_b.raking.designyear[nummer]) )
  
### Variance estimates for quantiles of induced distribution ####################################################################
  
  
  ### Variant A, Unweighted ###################################################
  
  
  # Number of eta for which q(P), min(P), max(P)
  which.eta.q <- which.min(abs(lips_a-lip_a.q[nummer]))
  which.eta.min <- which.min(lips_a)
  which.eta.max <- which.max(lips_a)
  
  # Values of eta for which q(P), min(P), max(P)
  eta.q <- scales_a[which.eta.q]
  eta.min <- scales_a[which.eta.min]
  eta.max <- scales_a[which.eta.max]
  
  # I(F)
  dat$z <- NA
  dat$zmin <- NA
  dat$zmax <- NA
  
  # I(F) for quantile
  # Calculate lip
    poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],probs=beta) 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=poverty.line.tmp] <- 1
    lip.tmp <- mean(dat$poor)
    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",which.eta.q,sep=".")],bw=bandwidth)
    fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=median(dat[,paste("equiv.inc",which.eta.q,sep=".")])] <- 1
    dat$z1 <- 1/N * (dat$poor - lip.tmp)
    dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
    dat$z <- dat$z1+dat$z2
  
  # I(F) for minimum
    # Calculate lip
    poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],probs=beta) 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=poverty.line.tmp] <- 1
    lip.tmp <- mean(dat$poor)
    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",which.eta.min,sep=".")],bw=bandwidth)
    fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=median(dat[,paste("equiv.inc",which.eta.min,sep=".")])] <- 1
    dat$z1 <- 1/N * (dat$poor - lip.tmp)
    dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
    dat$zmin <- dat$z1+dat$z2
  
  # I(F) for maximum
    # Calculate lip
    poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],probs=beta) 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=poverty.line.tmp] <- 1
    lip.tmp <- mean(dat$poor)
    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",which.eta.max,sep=".")],bw=bandwidth)
    fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=median(dat[,paste("equiv.inc",which.eta.max,sep=".")])] <- 1
    dat$z1 <- 1/N * (dat$poor - lip.tmp)
    dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
    dat$zmax <- dat$z1+dat$z2
  
  # Variance for random sample
  s_var_a.q[nummer] <- (var(dat$z)) * (N*(N-n))/n 
  s_var_a.min[nummer] <- (var(dat$zmin)) * (N*(N-n))/n
  s_var_a.max[nummer] <- (var(dat$zmax)) * (N*(N-n))/n
  # CIs
  CI_a.q[nummer,] <- c(lip_a.q[nummer]+1.96*sqrt(s_var_a.q[nummer]) , lip_a.q[nummer]-1.96*sqrt(s_var_a.q[nummer]))  
  CI_a.min[nummer,] <- c(lip_a.min[nummer]+1.96*sqrt(s_var_a.min[nummer]) , lip_a.min[nummer]-1.96*sqrt(s_var_a.min[nummer]))
  CI_a.max[nummer,] <- c(lip_a.max[nummer]+1.96*sqrt(s_var_a.max[nummer]) , lip_a.max[nummer]-1.96*sqrt(s_var_a.max[nummer]))
  CI_a.joint[nummer,] <- c(lip_a.max[nummer]+1.96*sqrt(s_var_a.max[nummer]),lip_a.min[nummer]-1.96*sqrt(s_var_a.min[nummer]))


  ### Variant A, Weighted #####################################################


  # Number of eta for which q(P), min(P), max(P)
  which.eta.q <- which.min(abs(lips_a.weighted-lip_a.weighted.q[nummer]))
  which.eta.min <- which.min(lips_a.weighted)
  which.eta.max <- which.max(lips_a.weighted)
  
  # Values of eta for which q(P), min(P), max(P)
  eta.q <- scales_a[which.eta.q]
  eta.min <- scales_a[which.eta.min]
  eta.max <- scales_a[which.eta.max]
  
  # I(F)
  dat$z <- NA
  dat$zmin <- NA
  dat$zmax <- NA
  
  # I(F) for quantile
    # Calculate lip
    poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=beta) 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=poverty.line.tmp] <- 1
    lip.tmp <- sum((dat$poor*dat$raking)/sum(dat$raking))
    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking/sum(dat$raking),bw=bandwidth)
    fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=0.5)] <- 1
    dat$z1 <- 1/N * (dat$poor - lip.tmp)
    dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
    dat$z <- dat$z1+dat$z2
  
  # I(F) for minimum
    # Calculate lip
    poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=beta) 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=poverty.line.tmp] <- 1
    lip.tmp <- sum((dat$poor*dat$raking)/sum(dat$raking))
    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",which.eta.min,sep=".")],bw=bandwidth,weights=dat$raking/sum(dat$raking))
    fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=0.5)] <- 1
    dat$z1 <- 1/N * (dat$poor - lip.tmp)
    dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
    dat$zmin <- dat$z1+dat$z2
  
  # I(F) for maximum
    # Calculate lip
    poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=beta) 
    dat$poor <- 0
    dat$poor[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=poverty.line.tmp] <- 1
    lip.tmp <- sum((dat$poor*dat$raking)/sum(dat$raking))
    # Density estimates (bandwidth according to Berger/Skinner after Silverman)
    bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
    tmp <- density(dat[,paste("equiv.inc",which.eta.max,sep=".")],bw=bandwidth,weights=dat$raking/sum(dat$raking))
    fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
    fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
    # Variance via influence function full sample
    dat$below.median <- 0
    dat$below.median[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=0.5)] <- 1
    dat$z1 <- 1/N * (dat$poor - lip.tmp)
    dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
    dat$zmax <- dat$z1+dat$z2
  
  # Correction of z's
  fit.raking <- lm(z~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$z <- fit.raking$residuals
  fit.raking <- lm(zmin~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$zmin <- fit.raking$residuals
  fit.raking <- lm(zmax~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$zmax <- fit.raking$residuals
  
  # Variance estimates (only raking)
  s_var_a.q.raking[nummer] <- (var(dat$z)) * (N*(N-n))/n 
  s_var_a.min.raking[nummer] <- (var(dat$zmin)) * (N*(N-n))/n
  s_var_a.max.raking[nummer] <- (var(dat$zmax)) * (N*(N-n))/n
  
  # CIs
  CI_a.q.raking[nummer,] <- c(lip_a.weighted.q[nummer]+1.96*sqrt(s_var_a.q.raking[nummer]) , lip_a.weighted.q[nummer]-1.96*sqrt(s_var_a.q.raking[nummer]))  
  CI_a.min.raking[nummer,] <- c(lip_a.weighted.min[nummer]+1.96*sqrt(s_var_a.min.raking[nummer]) , lip_a.weighted.min[nummer]-1.96*sqrt(s_var_a.min.raking[nummer]))
  CI_a.max.raking[nummer,] <- c(lip_a.weighted.max[nummer]+1.96*sqrt(s_var_a.max.raking[nummer]) , lip_a.weighted.max[nummer]-1.96*sqrt(s_var_a.max.raking[nummer]))
  CI_a.joint.raking[nummer,] <- c(lip_a.weighted.max[nummer]+1.96*sqrt(s_var_a.max.raking[nummer]),lip_a.weighted.min[nummer]-1.96*sqrt(s_var_a.min.raking[nummer]))
  
  # Variance estimate using design weights
  
  # Tmp Dummy (because some design weights=0)
  dat$use <- 0
  dat$use[dat$design>0] <- 1
  
  # Variance estimate  
  s_var_a.q.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  s_var_a.min.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmin[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  s_var_a.max.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmax[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  
  # CIs
  CI_a.q.raking.design[nummer,] <- c(lip_a.weighted.q[nummer]+1.96*sqrt(s_var_a.q.raking.design[nummer]) , lip_a.weighted.q[nummer]-1.96*sqrt(s_var_a.q.raking.design[nummer]))  
  CI_a.min.raking.design[nummer,] <- c(lip_a.weighted.min[nummer]+1.96*sqrt(s_var_a.min.raking.design[nummer]) , lip_a.weighted.min[nummer]-1.96*sqrt(s_var_a.min.raking.design[nummer]))
  CI_a.max.raking.design[nummer,] <- c(lip_a.weighted.max[nummer]+1.96*sqrt(s_var_a.max.raking.design[nummer]) , lip_a.weighted.max[nummer]-1.96*sqrt(s_var_a.max.raking.design[nummer]))
  CI_a.joint.raking.design[nummer,] <- c(lip_a.weighted.max[nummer]+1.96*sqrt(s_var_a.max.raking.design[nummer]),lip_a.weighted.min[nummer]-1.96*sqrt(s_var_a.min.raking.design[nummer]))
  
  # Variance estimate using synthetic design weights
  
  # Tmp dummy (because some synthetic design weights=0)
  dat$use <- 0
  dat$use[dat$designyear>0] <- 1
  
  # Variance estimate
  s_var_a.q.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  s_var_a.min.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmin[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  s_var_a.max.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmax[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  
  # CIs
  CI_a.q.raking.designyear[nummer,] <- c(lip_a.weighted.q[nummer]+1.96*sqrt(s_var_a.q.raking.designyear[nummer]) , lip_a.weighted.q[nummer]-1.96*sqrt(s_var_a.q.raking.designyear[nummer]))  
  CI_a.min.raking.designyear[nummer,] <- c(lip_a.weighted.min[nummer]+1.96*sqrt(s_var_a.min.raking.designyear[nummer]) , lip_a.weighted.min[nummer]-1.96*sqrt(s_var_a.min.raking.designyear[nummer]))
  CI_a.max.raking.designyear[nummer,] <- c(lip_a.weighted.max[nummer]+1.96*sqrt(s_var_a.max.raking.designyear[nummer]) , lip_a.weighted.max[nummer]-1.96*sqrt(s_var_a.max.raking.designyear[nummer]))
  CI_a.joint.raking.designyear[nummer,] <- c(lip_a.weighted.max[nummer]+1.96*sqrt(s_var_a.max.raking.designyear[nummer]),lip_a.weighted.min[nummer]-1.96*sqrt(s_var_a.min.raking.designyear[nummer]))
  
### Variance estimates for quantiles of induced distribution ####################################################################


  ### Variant B, Unweighted ###################################################

  
  # Number of eta for which q(P), min(P), max(P)
  which.eta.q <- which.min(abs(lips_b-lip_b.q[nummer]))
  which.eta.min <- which.min(lips_b)
  which.eta.max <- which.max(lips_b)
  
  # Values of eta for which q(P), min(P), max(P)
  eta.q <- scales_b[which.eta.q]
  eta.min <- scales_b[which.eta.min]
  eta.max <- scales_b[which.eta.max]
  
  # I(F)
  dat$z <- NA
  dat$zmin <- NA
  dat$zmax <- NA
  
  # I(F) for quantile
  # Calculate lip
  poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],probs=beta) 
  dat$poor <- 0
  dat$poor[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=poverty.line.tmp] <- 1
  lip.tmp <- mean(dat$poor)
  # Density estimates (bandwidth according to Berger/Skinner after Silverman)
  bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],probs=0.25))* n^(-0.2)
  tmp <- density(dat[,paste("equiv.inc",which.eta.q,sep=".")],bw=bandwidth)
  fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
  fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
  # Variance via influence function full sample
  dat$below.median <- 0
  dat$below.median[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=median(dat[,paste("equiv.inc",which.eta.q,sep=".")])] <- 1
  dat$z1 <- 1/N * (dat$poor - lip.tmp)
  dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
  dat$z <- dat$z1+dat$z2
  
  # I(F) for minimum
  # Calculate lip
  poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],probs=beta) 
  dat$poor <- 0
  dat$poor[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=poverty.line.tmp] <- 1
  lip.tmp <- mean(dat$poor)
  # Density estimates (bandwidth according to Berger/Skinner after Silverman)
  bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],probs=0.25))* n^(-0.2)
  tmp <- density(dat[,paste("equiv.inc",which.eta.min,sep=".")],bw=bandwidth)
  fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
  fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
  # Variance via influence function full sample
  dat$below.median <- 0
  dat$below.median[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=median(dat[,paste("equiv.inc",which.eta.min,sep=".")])] <- 1
  dat$z1 <- 1/N * (dat$poor - lip.tmp)
  dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
  dat$zmin <- dat$z1+dat$z2
  
  # I(F) for maximum
  # Calculate lip
  poverty.line.tmp <- alpha*quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],probs=beta) 
  dat$poor <- 0
  dat$poor[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=poverty.line.tmp] <- 1
  lip.tmp <- mean(dat$poor)
  # Density estimates (bandwidth according to Berger/Skinner after Silverman)
  bandwidth <- 0.79 * (quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],probs=0.75)-quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],probs=0.25))* n^(-0.2)
  tmp <- density(dat[,paste("equiv.inc",which.eta.max,sep=".")],bw=bandwidth)
  fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
  fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
  # Variance via influence function full sample
  dat$below.median <- 0
  dat$below.median[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=median(dat[,paste("equiv.inc",which.eta.max,sep=".")])] <- 1
  dat$z1 <- 1/N * (dat$poor - lip.tmp)
  dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
  dat$zmax <- dat$z1+dat$z2
  
  # Variance for random sample
  s_var_b.q[nummer] <- (var(dat$z)) * (N*(N-n))/n 
  s_var_b.min[nummer] <- (var(dat$zmin)) * (N*(N-n))/n
  s_var_b.max[nummer] <- (var(dat$zmax)) * (N*(N-n))/n
  # CIs
  CI_b.q[nummer,] <- c(lip_b.q[nummer]+1.96*sqrt(s_var_b.q[nummer]) , lip_b.q[nummer]-1.96*sqrt(s_var_b.q[nummer]))  
  CI_b.min[nummer,] <- c(lip_b.min[nummer]+1.96*sqrt(s_var_b.min[nummer]) , lip_b.min[nummer]-1.96*sqrt(s_var_b.min[nummer]))
  CI_b.max[nummer,] <- c(lip_b.max[nummer]+1.96*sqrt(s_var_b.max[nummer]) , lip_b.max[nummer]-1.96*sqrt(s_var_b.max[nummer]))
  CI_b.joint[nummer,] <- c(lip_b.max[nummer]+1.96*sqrt(s_var_b.max[nummer]),lip_b.min[nummer]-1.96*sqrt(s_var_b.min[nummer]))
  

  ### Variant B, Weighted #####################################################

  
  # Number of eta for which q(P), min(P), max(P)
  which.eta.q <- which.min(abs(lips_b.weighted-lip_b.weighted.q[nummer]))
  which.eta.min <- which.min(lips_b.weighted)
  which.eta.max <- which.max(lips_b.weighted)
  
  # Values of eta for which q(P), min(P), max(P)
  eta.q <- scales_b[which.eta.q]
  eta.min <- scales_b[which.eta.min]
  eta.max <- scales_b[which.eta.max]
  
  # I(F)
  dat$z <- NA
  dat$zmin <- NA
  dat$zmax <- NA
  
  # I(F) for quantile
  # Calculate lip
  poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=beta) 
  dat$poor <- 0
  dat$poor[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=poverty.line.tmp] <- 1
  lip.tmp <- sum((dat$poor*dat$raking)/sum(dat$raking))
  # Density estimates (bandwidth according to Berger/Skinner after Silverman)
  bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
  tmp <- density(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking/sum(dat$raking),bw=bandwidth)
  fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
  fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
  # Variance via influence function full sample
  dat$below.median <- 0
  dat$below.median[dat[,paste("equiv.inc",which.eta.q,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",which.eta.q,sep=".")],weights=dat$raking,probs=0.5)] <- 1
  dat$z1 <- 1/N * (dat$poor - lip.tmp)
  dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
  dat$z <- dat$z1+dat$z2
  
  # I(F) for minimum
  # Calculate lip
  poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=beta) 
  dat$poor <- 0
  dat$poor[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=poverty.line.tmp] <- 1
  lip.tmp <- sum((dat$poor*dat$raking)/sum(dat$raking))
  # Density estimates (bandwidth according to Berger/Skinner after Silverman)
  bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
  tmp <- density(dat[,paste("equiv.inc",which.eta.min,sep=".")],bw=bandwidth,weights=dat$raking/sum(dat$raking))
  fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
  fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
  # Variance via influence function full sample
  dat$below.median <- 0
  dat$below.median[dat[,paste("equiv.inc",which.eta.min,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",which.eta.min,sep=".")],weights=dat$raking,probs=0.5)] <- 1
  dat$z1 <- 1/N * (dat$poor - lip.tmp)
  dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
  dat$zmin <- dat$z1+dat$z2
  
  # I(F) for maximum
  # Calculate lip
  poverty.line.tmp <- alpha*wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=beta) 
  dat$poor <- 0
  dat$poor[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=poverty.line.tmp] <- 1
  lip.tmp <- sum((dat$poor*dat$raking)/sum(dat$raking))
  # Density estimates (bandwidth according to Berger/Skinner after Silverman)
  bandwidth <- 0.79 * (wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=0.75)-wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=0.25))* n^(-0.2)
  tmp <- density(dat[,paste("equiv.inc",which.eta.max,sep=".")],bw=bandwidth,weights=dat$raking/sum(dat$raking))
  fe1 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp))]
  fe2 <- tmp$y[which.min(abs(tmp$x-poverty.line.tmp/alpha))]
  # Variance via influence function full sample
  dat$below.median <- 0
  dat$below.median[dat[,paste("equiv.inc",which.eta.max,sep=".")]<=wtd.quantile(dat[,paste("equiv.inc",which.eta.max,sep=".")],weights=dat$raking,probs=0.5)] <- 1
  dat$z1 <- 1/N * (dat$poor - lip.tmp)
  dat$z2 <- -alpha * 1/N * (fe1/fe2) * (dat$below.median-beta)
  dat$zmax <- dat$z1+dat$z2
  
  # Correction of z's
  fit.raking <- lm(z~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$z <- fit.raking$residuals
  fit.raking <- lm(zmin~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$zmin <- fit.raking$residuals
  fit.raking <- lm(zmax~factor(family.size)+factor(bula)+factor(owner)+factor(comsize),data=dat,weights=dat$raking)
  dat$zmax <- fit.raking$residuals
  
  # Variance estimates (only raking)
  s_var_b.q.raking[nummer] <- (var(dat$z)) * (N*(N-n))/n 
  s_var_b.min.raking[nummer] <- (var(dat$zmin)) * (N*(N-n))/n
  s_var_b.max.raking[nummer] <- (var(dat$zmax)) * (N*(N-n))/n
  
  # CIs
  CI_b.q.raking[nummer,] <- c(lip_b.weighted.q[nummer]+1.96*sqrt(s_var_b.q.raking[nummer]) , lip_b.weighted.q[nummer]-1.96*sqrt(s_var_b.q.raking[nummer]))  
  CI_b.min.raking[nummer,] <- c(lip_b.weighted.min[nummer]+1.96*sqrt(s_var_b.min.raking[nummer]) , lip_b.weighted.min[nummer]-1.96*sqrt(s_var_b.min.raking[nummer]))
  CI_b.max.raking[nummer,] <- c(lip_b.weighted.max[nummer]+1.96*sqrt(s_var_b.max.raking[nummer]) , lip_b.weighted.max[nummer]-1.96*sqrt(s_var_b.max.raking[nummer]))
  CI_b.joint.raking[nummer,] <- c(lip_b.weighted.max[nummer]+1.96*sqrt(s_var_b.max.raking[nummer]),lip_b.weighted.min[nummer]-1.96*sqrt(s_var_b.min.raking[nummer]))
  
  # Variance estimate using design weights
  
  # Tmp Dummy (because some design weights=0)
  dat$use <- 0
  dat$use[dat$design>0] <- 1
  
  # Variance estimate  
  s_var_b.q.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  s_var_b.min.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmin[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  s_var_b.max.raking.design[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmax[dat$use==1],VecPk.s=1/dat$design[dat$use==1])
  
  # CIs
  CI_b.q.raking.design[nummer,] <- c(lip_b.weighted.q[nummer]+1.96*sqrt(s_var_b.q.raking.design[nummer]) , lip_b.weighted.q[nummer]-1.96*sqrt(s_var_b.q.raking.design[nummer]))  
  CI_b.min.raking.design[nummer,] <- c(lip_b.weighted.min[nummer]+1.96*sqrt(s_var_b.min.raking.design[nummer]) , lip_b.weighted.min[nummer]-1.96*sqrt(s_var_b.min.raking.design[nummer]))
  CI_b.max.raking.design[nummer,] <- c(lip_b.weighted.max[nummer]+1.96*sqrt(s_var_b.max.raking.design[nummer]) , lip_b.weighted.max[nummer]-1.96*sqrt(s_var_b.max.raking.design[nummer]))
  CI_b.joint.raking.design[nummer,] <- c(lip_b.weighted.max[nummer]+1.96*sqrt(s_var_b.max.raking.design[nummer]),lip_b.weighted.min[nummer]-1.96*sqrt(s_var_b.min.raking.design[nummer]))
  
  # Variance estimate using synthetic design weights
  
  # Tmp dummy (because some synthetic design weights=0)
  dat$use <- 0
  dat$use[dat$designyear>0] <- 1
  
  # Variance estimate
  s_var_b.q.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$z[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  s_var_b.min.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmin[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  s_var_b.max.raking.designyear[nummer] <- VE.Hajek.Total.NHT(VecY.s=dat$zmax[dat$use==1],VecPk.s=1/dat$designyear[dat$use==1])
  
  # CIs
  CI_b.q.raking.designyear[nummer,] <- c(lip_b.weighted.q[nummer]+1.96*sqrt(s_var_b.q.raking.designyear[nummer]) , lip_b.weighted.q[nummer]-1.96*sqrt(s_var_b.q.raking.designyear[nummer]))  
  CI_b.min.raking.designyear[nummer,] <- c(lip_b.weighted.min[nummer]+1.96*sqrt(s_var_b.min.raking.designyear[nummer]) , lip_b.weighted.min[nummer]-1.96*sqrt(s_var_b.min.raking.designyear[nummer]))
  CI_b.max.raking.designyear[nummer,] <- c(lip_b.weighted.max[nummer]+1.96*sqrt(s_var_b.max.raking.designyear[nummer]) , lip_b.weighted.max[nummer]-1.96*sqrt(s_var_b.max.raking.designyear[nummer]))
  CI_b.joint.raking.designyear[nummer,] <- c(lip_b.weighted.max[nummer]+1.96*sqrt(s_var_b.max.raking.designyear[nummer]),lip_b.weighted.min[nummer]-1.96*sqrt(s_var_b.min.raking.designyear[nummer]))
  
### End analyis loop ############################################################################################################

  }

### Save results ################################################################################################################

  datei <- "/home/christian/Dokumente/Equiv Variance/Ergebnisse/example.rda"
  save(list=ls(),file=datei)


#################################################################################################################################
#################################################################################################################################